<?php
if (!defined('ABSPATH')) exit;

class GNews_Sitemap_Generator {

    // ✅ Run on plugin activation
    public static function activate() {
        self::generate_sitemap();
    }

    // ✅ Cleanup on plugin deactivation
    public static function deactivate() {
        $file = ABSPATH . get_option('gnews_sitemap_name', 'google-news-sitemap.xml');
        if (file_exists($file)) unlink($file);
    }

    // ✅ Generate the Google News Sitemap
    public static function generate_sitemap() {
        $categories = (array) get_option('gnews_sitemap_categories', []);
        $days = get_option('gnews_sitemap_days', 7);
        $filename = sanitize_file_name(get_option('gnews_sitemap_name', 'google-news-sitemap.xml'));

        // ✅ Fetch publication name (or fallback to site title)
        $publication_name = trim(get_option('gnews_publication_name', ''));
        if (empty($publication_name)) {
            $publication_name = get_bloginfo('name');
        }

        // ✅ Detect and normalize site language
        $site_lang = function_exists('get_locale') ? get_locale() : 'en_US';

        // Polylang or WPML language override
        if (function_exists('pll_current_language')) {
            $pll_locale = pll_current_language('locale');
            if (!empty($pll_locale)) {
                $site_lang = $pll_locale;
            }
        } elseif (defined('ICL_LANGUAGE_CODE')) {
            $wpml_lang = apply_filters('wpml_current_language', NULL);
            if (!empty($wpml_lang)) {
                $site_lang = str_replace('-', '_', $wpml_lang) . '_';
            }
        }

        // Normalize to Google News format (en-US, es-ES, etc.)
        $site_lang = str_replace('_', '-', strtolower($site_lang));
        if (strlen($site_lang) == 2) {
            $site_lang = $site_lang . '-' . strtoupper($site_lang);
        }

        // ✅ Filter posts by selected categories and recent days
        $date_query = [
            'after' => date('Y-m-d', strtotime("-{$days} days"))
        ];

        $args = [
            'post_type'      => 'post',
            'posts_per_page' => -1,
            'date_query'     => [$date_query],
            'category__in'   => $categories,
            'post_status'    => 'publish'
        ];

        $posts = get_posts($args);

        // ✅ Start XML output
        $sitemap  = '<?xml version="1.0" encoding="UTF-8"?>' . PHP_EOL;
        $sitemap .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" ';
        $sitemap .= 'xmlns:news="http://www.google.com/schemas/sitemap-news/0.9">' . PHP_EOL;

        foreach ($posts as $post) {
            setup_postdata($post);

            // Collect categories and tags for <news:keywords>
            $cat_names = [];
            $tag_names = [];

            $post_cats = get_the_category($post->ID);
            $post_tags = get_the_tags($post->ID);

            if ($post_cats) {
                foreach ($post_cats as $cat_obj) {
                    $cat_names[] = $cat_obj->name;
                }
            }

            if ($post_tags) {
                foreach ($post_tags as $tag_obj) {
                    $tag_names[] = $tag_obj->name;
                }
            }

            $keywords = array_merge($cat_names, $tag_names);
            $keywords_str = esc_html(implode(', ', $keywords));

            // ✅ Build XML item
            $sitemap .= "<url>" . PHP_EOL;
            $sitemap .= "  <loc>" . esc_url(get_permalink($post->ID)) . "</loc>" . PHP_EOL;
            $sitemap .= "  <news:news>" . PHP_EOL;
            $sitemap .= "    <news:publication>" . PHP_EOL;
            $sitemap .= "      <news:name>" . esc_html($publication_name) . "</news:name>" . PHP_EOL;
            $sitemap .= "      <news:language>" . esc_html($site_lang) . "</news:language>" . PHP_EOL;
            $sitemap .= "    </news:publication>" . PHP_EOL;
            $sitemap .= "    <news:publication_date>" . get_the_date('c', $post->ID) . "</news:publication_date>" . PHP_EOL;
            $sitemap .= "    <news:title>" . esc_html(get_the_title($post->ID)) . "</news:title>" . PHP_EOL;
            if (!empty($keywords_str)) {
                $sitemap .= "    <news:keywords>" . $keywords_str . "</news:keywords>" . PHP_EOL;
            }
            $sitemap .= "  </news:news>" . PHP_EOL;
            $sitemap .= "</url>" . PHP_EOL;
        }

        wp_reset_postdata();

        $sitemap .= '</urlset>';

        // ✅ Save sitemap file to root directory
        $file_path = ABSPATH . $filename;
        file_put_contents($file_path, $sitemap);
    }
}

// ✅ Automatically regenerate sitemap whenever a post is published
add_action('publish_post', function ($post_id) {
    if (class_exists('GNews_Sitemap_Generator')) {
        GNews_Sitemap_Generator::generate_sitemap();
    }
});

// ✅ Optional hourly cron job (keeps sitemap fresh)
if (!wp_next_scheduled('gnews_sitemap_cron_refresh')) {
    wp_schedule_event(time(), 'hourly', 'gnews_sitemap_cron_refresh');
}
add_action('gnews_sitemap_cron_refresh', ['GNews_Sitemap_Generator', 'generate_sitemap']);
