<?php
if (!defined('ABSPATH')) exit;

class GNews_Sitemap_Admin {

    public function __construct() {
        add_action('admin_menu', [$this, 'add_settings_page']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_post_gnews_generate_sitemap', [$this, 'generate_sitemap_now']);
    }

    public function add_settings_page() {
        add_options_page(
            'Google News Sitemap',
            'Google News Sitemap',
            'manage_options',
            'google-news-sitemap',
            [$this, 'settings_page_html']
        );
    }

    public function register_settings() {
        register_setting('gnews_sitemap_options', 'gnews_sitemap_categories'); // ✅ multiple categories
        register_setting('gnews_sitemap_options', 'gnews_sitemap_days');
        register_setting('gnews_sitemap_options', 'gnews_sitemap_name');
        register_setting('gnews_sitemap_options', 'gnews_publication_name');
    }

    public function settings_page_html() {
        if (!current_user_can('manage_options')) return;

        $categories = get_categories(['hide_empty' => false]);
        $selected_cats = (array) get_option('gnews_sitemap_categories', []);
        $selected_days = get_option('gnews_sitemap_days', 7);
        $sitemap_name = get_option('gnews_sitemap_name', 'google-news-sitemap.xml');
        $publication_name = get_option('gnews_publication_name', get_bloginfo('name'));
        $sitemap_url = home_url('/' . $sitemap_name);
        $sitemap_path = ABSPATH . $sitemap_name;

        if (isset($_GET['generated']) && $_GET['generated'] == 'true') {
            echo '<div class="updated notice"><p>✅ Sitemap generated successfully! 
            <a href="' . esc_url($sitemap_url) . '" target="_blank">View Sitemap</a></p></div>';
        }
        ?>

        <div class="wrap">
            <h1>Google News Sitemap Settings</h1>

            <form method="post" action="options.php">
                <?php settings_fields('gnews_sitemap_options'); ?>
                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">Google News Publication Name</th>
                        <td>
                            <input type="text" name="gnews_publication_name" value="<?php echo esc_attr($publication_name); ?>" class="regular-text" />
                            <p class="description">
                                This should match your publication name registered in Google News.<br>
                                Example: <code>Indo-American Indian American News, Voices, Insights!</code>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Select Categories</th>
                        <td>
                            <?php foreach ($categories as $cat): ?>
                                <label>
                                    <input type="checkbox" name="gnews_sitemap_categories[]" value="<?php echo esc_attr($cat->term_id); ?>" 
                                        <?php checked(in_array($cat->term_id, $selected_cats)); ?> />
                                    <?php echo esc_html($cat->name); ?>
                                </label><br>
                            <?php endforeach; ?>
                            <p class="description">Select one or more categories to include in the sitemap.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Number of Days (up to 30)</th>
                        <td>
                            <input type="number" name="gnews_sitemap_days" value="<?php echo esc_attr($selected_days); ?>" min="1" max="30" />
                            <p class="description">Only include posts published within this many days.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">Sitemap File Name</th>
                        <td>
                            <input type="text" name="gnews_sitemap_name" value="<?php echo esc_attr($sitemap_name); ?>" placeholder="google-news-sitemap.xml" />
                            <p class="description">Enter a file name ending with <code>.xml</code> (e.g. <code>mynews.xml</code>).</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button('Save Settings'); ?>
            </form>

            <form method="post" action="<?php echo admin_url('admin-post.php'); ?>" style="margin-top: 20px;">
                <?php wp_nonce_field('gnews_generate_sitemap_nonce', 'gnews_generate_sitemap_nonce_field'); ?>
                <input type="hidden" name="action" value="gnews_generate_sitemap">
                <?php submit_button('Save Settings & Generate Sitemap Now', 'primary', 'generate_sitemap'); ?>
            </form>

            <hr>

            <h2>📄 Current Sitemap</h2>
            <?php if (file_exists($sitemap_path)) : ?>
                <p>Your active sitemap file is: 
                    <a href="<?php echo esc_url($sitemap_url); ?>" target="_blank">
                        <?php echo esc_html($sitemap_url); ?>
                    </a>
                </p>
                <p>Last updated: <strong><?php echo date("F j, Y, g:i a", filemtime($sitemap_path)); ?></strong></p>
            <?php else : ?>
                <p>No sitemap generated yet. Click “Save Settings & Generate Sitemap Now” to create one.</p>
            <?php endif; ?>
        </div>
        <?php
    }

    public function generate_sitemap_now() {
        if (!current_user_can('manage_options')) return;
        if (!isset($_POST['gnews_generate_sitemap_nonce_field']) || !wp_verify_nonce($_POST['gnews_generate_sitemap_nonce_field'], 'gnews_generate_sitemap_nonce')) return;

        GNews_Sitemap_Generator::generate_sitemap();

        wp_redirect(add_query_arg('generated', 'true', admin_url('options-general.php?page=google-news-sitemap')));
        exit;
    }
}

new GNews_Sitemap_Admin();
